<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\UserRequest;
use App\Models\Branch;
use App\Models\ModelHasPermission;
use App\Models\ModulePermission;
use App\Models\Permission;
use App\Models\UploadFile;
use App\Models\User;
use App\Services\QueryService;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;

class UserController extends Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->middleware('permission:user-view', ['only' => ['index']]);
        $this->middleware('permission:user-create', ['only' => ['onCreate', 'onSave']]);
        $this->middleware('permission:user-update', ['only' => ['onCreate', 'onSave', 'onUpdateStatus', 'onRestore', 'onChangePassword', 'onSavePassword', 'setPermission', 'savePermission']]);
        $this->middleware('permission:user-delete', ['only' => ['onDelete']]);
    }

    public function login()
    {
        if (Auth::check()) {
            return redirect()->route('admin-dashboard');
        }
        return view("admin::auth.sign-in");
    }

    public function forgotPassword()
    {
        return view("admin::auth.forgot-password");
    }

    public function index(Request $req)
    {
        if (!$req->id) {
            return redirect()->route('admin-user-list', 1);
        }
        $data['data'] = User::when(filled(request('keyword')), function ($q) {
                $q->where(function ($q) {
                    $q->where('username', 'like', '%' . request('keyword') . '%');
                    $q->orWhere('phone', 'like', '%' . request('keyword') . '%');
                    $q->orWhere('email', 'like', '%' . request('keyword') . '%');
                });
            })
            ->whereNotIn('role', [$this->isRoleSuperAdmin])
            ->when(request('role'), function ($q) {
                $q->where('role', request('role'));
            })            
            ->when(request('branch'), function ($q) {
                $q->where('branch_id', request('branch'));
            })
            ->where("status", $req->id)
            ->when($this->userBranch, function ($q) {
                $q->where('branch_id', $this->userBranch->id);
            })
            ->orderByDesc("created_at")
            ->paginate(50);
        

        $data['branches']   = Branch::whereStatus($this->active)
            ->orderByDesc("id")
            ->get();

        return view("admin::pages.user.index", $data);
    }

    public function viewTrash(Request $req)
    {
        $data['id'] = $req->id;
        $data['data'] = User::onlyTrashed()
            ->when(filled(request('keyword')), function ($q) {
                $q->where('name', 'like', '%' . request('keyword') . '%')
                    ->orWhere('phone', 'like', '%' . request('keyword') . '%')
                    ->orWhere('email', 'like', '%' . request('keyword') . '%');
            })
            ->whereNotIn('role', [$this->isRoleSuperAdmin])
            ->when(request('role'), function ($q) {
                $q->where('role', request('role'));
            })
            ->when(request('branch'), function ($q) {
                $q->where('branch_id', request('branch'));
            })
            ->when($this->userBranch, function ($q) {
                $q->where('branch_id', $this->userBranch->id);
            })
            ->orderBy("created_at", "desc")
            ->paginate(10);

        $data['branches']   = Branch::whereStatus($this->active)
                                ->orderByDesc("id")
                                ->get();

        return view("admin::pages.user.index", $data);
    }

    public function onCreate(Request $req)
    {
        $data["data"]       = User::find($req->id);
        $data['branches']   = Branch::whereStatus($this->active)
                                ->orderByDesc("id")
                                ->get();

        return view("admin::pages.user.create", $data);
    }

    public function onSave(UserRequest $req)
    { 
        $id = $req->id;
        $item = [
            "branch_id"         => $req->branch_id,
            "username"          => $req->name,
            "email"             => $req->email,
            "phone"             => $req->phone,
            "status"            => $req->status,
            "profile"           => $req->image ?? $req->tmp_file ?? null,
            "remember_token"    => $req->_token,
            "role"              => $req->is_teacher ? $this->isRoleTeacher : $this->isRoleAdmin,
        ];
        
        $status = "Create success.";
        try {
            if (!$id) {
                $item["password"] = bcrypt($req->password);
                User::create($item);
            } else {
                $user = User::find($id);

                // check if role is super admin and user, can't update
                if ($user->role == $this->isRoleSuperAdmin || $user->role == $this->isRoleUser) {
                    Session::flash('warning', 'Can\'t update!');
                    return redirect()->back();
                }

                $user->update($item);
                $status = "Update success.";
            }
            Session::flash("success", $status);
            return redirect()->route("admin-user-list", 1);
        } catch (Exception $error) {
            Session::flash('warning', 'Create unsuccess!');
            return redirect()->back();
        }
    }

    public function onChangePassword(Request $req)
    {
        $user = User::where('id', $req->id)->first();
        if ($user->role == $this->isRoleSuperAdmin) {
            return redirect()->route("admin-user-list", 1);
        }
        return view("admin::pages.user.change-password", ['data' => $user]);
    }

    public function onSavePassword(Request $req)
    {
        $item = [
            "password" => bcrypt($req->password),
        ];
        try {
            $user = User::find($req->id);
            $user->update($item);
            $status = "change password success";
            Session::flash("success", $status);
        } catch (Exception $error) {
            Session::flash("warning", "change password unsuccess");
        }
        return redirect()->route("admin-user-list", 1);
    }

    public function onUpdateStatus(Request $req)
    {
        $status = true;
        $item = [
            "status" => $req->status,
        ];
        try {
            $status = $req->status == 2 ? "Disable successful!" : "Enable successful!";
            User::where("id", $req->id)->update($item);
            Session::flash("success", $status);
        } catch (Exception $error) {
            Session::flash('warning', 'Create unsuccess!');
        }
        return redirect()->back();
    }

    public function onDelete(Request $req)
    {
        $status = "Delete successful!";
        try {
            if ($req->to_trash) {
                User::find($req->id)->delete();
            }
            $status = true;
        } catch (Exception $error) {
            $status = "Delete unsuccess!";
        }
        Session::flash("success", $status);
        return redirect()->back();
    }

    public function onRestore(Request $req)
    {
        $status = "Restore successful!";
        try {
            User::withTrashed()->find($req->id)->restore();
            Session::flash("success", $status);
        } catch (Exception $error) {
            $status = "Restore unsuccess!";
            Session::flash("warning", $status);
        }
        return redirect()->back();
    }

    public function setPermission()
    {
        // check user can't update yourself and super admin
        $user = User::find(request("id"));
        if ($user->role == "super_admin" || $user->id == Auth::user()->id) {
            return redirect()->back();
        }
        $data["user"] = User::find(request('id'));
        $data['ModulPermission'] = ModulePermission::select('parent_id')->groupBy('parent_id')->orderBy('sort_no')->get();
        $data['permission'] = $data["user"]->ModelHasPermission;
        return view("admin::pages.user.permission", $data);
    }

    public function savePermission(Request $req)
    {
        $req->validate([
            "permission" => "required",
        ], [
            "permission.required" => "Permission required",
        ]);
        if (!$req->permission) {
            return redirect()->back();
        }
        DB::beginTransaction();
        try {
            $data = User::find($req->id);
            $permissions = Permission::pluck('name')->toArray();
            $revoke = array_diff($permissions, $req->permission);
            $data->givePermissionTo($req->permission);
            $data->revokePermissionTo($revoke);
            DB::commit();
            Session::flash("success", 'Set permission successful!');
            return redirect()->route("admin-user-list", 1);
            
        } catch (Exception $error) {
            DB::rollback();
            $status = "Permission unsuccess!";
            Session::flash("warning", $status);
            return redirect()->back();
        }
    }
}