<?php

namespace App\Http\Controllers\Admin;

use App\Exports\StudentsExport;
use App\Http\Controllers\Controller;
use App\Models\Attendance;
use App\Models\Classes;
use App\Models\ClassLevel;
use App\Models\ClassOccupation;
use App\Models\ClassStatus;
use App\Models\ClassStructureProgram;
use App\Models\ClassType;
use App\Models\Shift;
use App\Models\Staff;
use App\Models\StructureProgram;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use Exception;
use App\Models\Student;
use App\Models\EducationLevel;
use App\Models\Lesson;
use Maatwebsite\Excel\Facades\Excel;

class StudentController extends Controller
{
    protected $layout = 'admin::pages.student.';
    public function __construct()
    {
        $this->middleware('permission:student-view', ['only' => ['index']]);
        $this->middleware('permission:student-create', ['only' => ['onCreate', 'onSave']]);
        $this->middleware('permission:student-update', ['only' => ['onCreate', 'onSave', 'onUpdateStatus', 'onUpdateRequestApprovalStatus']]);
        $this->middleware('permission:student-view-info', ['only' => ['onView']]);
        $this->middleware('permission:student-assign', ['only' => ['onAssignClass']]);
    }
    public function index(Request $req)
    {
        if (!$req->id) {
            return redirect()->route('admin-student-list', 1);
        }
        $data['education_levels'] = EducationLevel::where('status', 1)->get();
        $data['data'] = Student::with('classes.class_status', 'classes.class_structure_program.structure_program')
            ->when(filled(request('keyword')), function ($q) {
                $q->where(function ($q) {
                    $q->where('name', 'like', '%' . request('keyword') . '%');
                });
            })
            ->when(filled(request('education_level')), function ($q) {
                $q->where('education_level_id', request('education_level'));
            })
            ->when($req->id, function($q) use ($req){
                if($req->id != 'request_approval'){
                    $q->where('status', $req->id);
                    $q->where('approval_status', 'approved');
                }elseif($req->id == 'request_approval'){
                    $q->whereIn('approval_status', ['pending', 'rejected']);
                }
            })
            ->orderByDesc("id")
            ->paginate(25); 
        foreach ($data['data'] as $item) {
            foreach ($item->classes as $class) {
                $class->class_structure_program->structure_program->practice_hour = Lesson::whereIn('id', $class?->class_structure_program?->structure_program?->lesson_id)
                    ->sum('practice_hour');
                $class->class_structure_program->structure_program->theory_hour = Lesson::whereIn('id', $class?->class_structure_program?->structure_program?->lesson_id)
                    ->sum('theory_hour');
                $class->class_structure_program->structure_program->attend_amount = Attendance::where('class_id', $class->id)
                    ->whereIn('lesson_id', $class?->class_structure_program?->structure_program?->lesson_id)
                    ->count();
            }
        }
        return view($this->layout . 'index', $data);
    }
    public function onCreate(Request $req)
    {
        $data["data"] = Student::find($req->id);
        $data['education_level'] = EducationLevel::where('status', 1)->get();
        return view($this->layout . 'create', $data);
    }

    public function onSave(Request $req)
    {
        $id = $req->id;
        $item = [
            "education_level_id" => $req->education_level_id,
            "name_latin" => $req->name_latin,
            "name_khmer" => $req->name_khmer,
            "dob" => $req->dob,
            "id_card" =>  $req->id_card,
            "sid" =>  $req->sid,
            "profile" =>  $req->image ?? $req->tmp_file ?? null,
            "phone" => $req->phone,
            "phone_with_telegram" => $req->phone_with_telegram,
            "gender" => $req->gender,
            "email" => $req->email,
            "address_khmer" => $req->address_khmer,
            "address_latin" => $req->address_latin,
            "noted" => $req->noted,
            "status" => $req->status ?? 1,
            "current_working_place" => $req->current_working_place,
            "working_experience" => $req->working_experience,
        ];

        $req->validate([
            "email"                 => "nullable|unique:students,email" . ($id ? ",$id" : ''),
            "phone"                 => "required|unique:students,phone" . ($id ? ",$id" : ''),
            "phone_with_telegram"   => "required|unique:students,phone_with_telegram" . ($id ? ",$id" : ''),
            "name_khmer"            => "required",
            "name_latin"            => "required",
            "dob"                   => "required",
            "sid"                   => "required|unique:students,sid" . ($id ? ",$id" : ''),
            "address_khmer"         => "required",
            "education_level_id"    => "required",
            "gender"                => "required",
        ], [
            "email.unique"                  => "Email already exist",
            "phone.unique"                  => "Phone number already exist",
            "phone.required"                => "Please input this field",
            "phone_with_telegram.unique"    => "Telegram phone number already exist",
            "phone_with_telegram.required"  => "Please input this field",
            "name_khmer.required"           => "Please input this field",
            "name_latin.required"           => "Please input this field",
            "dob.required"                  => "Please input this field",
            "sid.required"                  => "Please input this field",
            "sid.unique"                    => "Student ID already exists",
            "address_khmer.required"        => "Please input this field",
            "education_level_id.required"   => "Please input this field",
            "gender.required"               => "Please input this field",
        ]);
        $status = "Create success.";
        DB::beginTransaction();
        try {
            if (!$id) {
                Student::create($item);
            } else {
                Student::find($id)->update($item);
                $status = "Update success.";
            }
            DB::commit();
            Session::flash("success", $status);
            return redirect()->route("admin-student-list", 1);
        } catch (Exception $error) {
            dd($error);
            DB::rollback();
            Session::flash('warning', 'Create unsuccess!');
            return redirect()->back();
        }
    }
    public function onUpdateStatus(Request $req)
    {
        $status = true;
        $item = [
            "status" => $req->status,
        ];
        DB::beginTransaction();
        try {
            $status = $req->status == 2 ? "Disable successful!" : "Enable successful!";
            Student::where("id", $req->id)->update($item);
            DB::commit();
            Session::flash("success", $status);
        } catch (Exception $error) {
            DB::rollback();
            Session::flash('warning', 'Create unsuccess!');
        }
        return redirect()->back();
    }

    public function onAssignClass(Request $req)
    {

        $data['data'] = Student::find($req->id);
        $classes = Classes::whereIn('class_status_id', [1, 2])
            ->orderByDesc("id")->get();
        $data['classes'] = [];
        foreach ($classes as $item) {
            if ($item->class_structure_program) {
                if (count($item->students) > 0) {
                    $count_student = 0;
                    foreach ($item->students as $student) {
                        if ($student->id != $data['data']->id) {
                            $count_student += 1;
                        }
                        if ($count_student == count($item->students)) {
                            $data['classes'][] = $item;
                        }
                    }
                } else {
                    $data['classes'][] = $item;
                }
            }
        }
        return view($this->layout . 'assign-class', $data);
    }

    public function onSaveAssignClass(Request $req)
    {
        $data = Student::find($req->id);
        DB::beginTransaction();
        try {
            $data->classes()->sync($req->class_id, false);
            DB::commit();
            Session::flash("success", 'Assign class success');
            return redirect()->route("admin-student-list", 1);
        } catch (Exception $error) {
            DB::rollback();
            Session::flash('warning', 'Assign class unsuccess!');
            return redirect()->back();
        }
    }

    public function onView()
    {
        $data['data'] = Student::findOrFail(request('id'));
        return view($this->layout . 'view-info', $data);
    }

    public function exportExcel(){
        $data = Student::with('education_level')
        ->when(filled(request('keyword')), function ($q) {
            $q->where(function ($q) {
                $q->where('name', 'like', '%' . request('keyword') . '%');
            });
        })
        ->when(filled(request('education_level')), function ($q) {
            $q->where('education_level_id', request('education_level'));
        })
        ->where('status', request('id'))
        ->orderByDesc("id")
        ->get(); 

        return Excel::download(new StudentsExport($data), 'export_lice_students.xlsx');
    }

    public function onUpdateRequestApprovalStatus(Request $req){
        try {
            $item = [
                "approval_status" => $req->approval_status,
            ];
            DB::beginTransaction();
            $status = "Update approval status to ".$req->approval_status." successful!";
            Student::where("id", $req->id)->update($item);
            DB::commit();
            Session::flash("success", $status);
        } catch (Exception $error) {
            DB::rollback();
            Session::flash('warning', 'Update unsuccess!');
        }
        return redirect()->back();
    }
}
