<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Attendance;
use App\Models\ClassStructureProgram;
use App\Models\Lesson;
use App\Models\Shift;
use App\Models\Staff;
use App\Models\User;
use App\Models\Student;
use App\Models\StructureProgram;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use Exception;
use App\Models\Classes;
use App\Models\Group;
use App\Models\ClassStatus;
use App\Models\ClassLevel;
use App\Models\ClassType;
use App\Models\StudentInternPlace;
use Carbon\Carbon;

class ClassController extends Controller
{
    protected $layout = 'admin::pages.class.';
    public function __construct()
    {
        parent::__construct();
        $this->middleware('permission:class-view', ['only' => ['index']]);
        $this->middleware('permission:class-create', ['only' => ['onCreate', 'onSave']]);
        $this->middleware('permission:class-update', ['only' => ['onCreate', 'onSave', 'onUpdateStatus']]);
        $this->middleware('permission:class-view-info', ['only' => ['onView']]);
        $this->middleware('permission:class-study-material', ['only' => ['studyMaterial', 'studentCard']]);
        $this->middleware('permission:class-assign', ['only' => ['onAssignStructured', 'onSaveAssignStructured']]);
    }
    public function index(Request $req)
    {
        if (!$req->id) {
            return redirect()->route('admin-class-list', 1);
        }
        $data['class_type'] = ClassType::where('status', 1)->get();
        $data['class_level'] = ClassLevel::where('status', 1)->get();
        $data['class_status'] = ClassStatus::where('status', 1)->get();
        $data['shift'] = Shift::where('status', 1)->get();
        $data['staff'] = Staff::where('status', 1)->get();
        $data['data'] = Classes::when(filled(request('keyword')), function ($q) {
            $q->where(function ($q) {
                $q->where('class_code', 'like', '%' . request('keyword') . '%');
            });
        })
            ->when($this->userBranch, function ($q) {
                $q->whereHas('user', function ($q) {
                    $q->where('branch_id', $this->userBranch->id);
                });
            })
            ->when(request('class_cost'), function($q){
                $q->where(function($q){
                    if(request('class_cost') == 'free'){
                        $q->where('class_cost', 0); 
                    }
                    if(request('class_cost') == 'cost'){
                        $q->where('class_cost', '!=' ,0); 
                    }
                });
            })
            ->when(request('supplies_cost'), function($q){
                $q->where(function($q){
                    if(request('supplies_cost') == 'free'){
                        $q->where('supplies_cost', 0); 
                    }
                    if(request('supplies_cost') == 'cost'){
                        $q->where('supplies_cost', '!=' ,0); 
                    }
                });
            })
            ->orderByDesc("id")
            ->paginate(50);
        return view($this->layout . 'index', $data);
    }
    public function onCreate(Request $req)
    {
        $data["data"] = Classes::find($req->id);
        $data['class_type'] = ClassType::where('status', 1)->get();
        $data['class_level'] = ClassLevel::where('status', 1)->get();
        $data['class_status'] = ClassStatus::where('status', 1)->get();
        $data['shift'] = Shift::where('status', 1)->get();
        $data['staff'] = Staff::where('status', 1)->get();
        return view($this->layout . 'create', $data);
    }

    public function onSave(Request $req)
    {
        $id = $req->id;
        $item = [
            "class_status_id"       => $req->class_status_id,
            "class_level_id"        => $req->class_level_id,
            "class_type_id"         => $req->class_type_id,
            "staff_id"              => $req->staff_id,
            "shift_id"              => $req->shift_id,
            "class_code"            => $req->class_code,
            "generation"            => $req->generation,
            "group"                 => $req->group,
            "period_duration"       => $req->period_duration,
            "start_date"            => $req->start_date,
            "end_date"              => $req->end_date,
            "url_google_sheet_edit" => $req->url_google_sheet_edit,
            "url_google_sheet_view" => $req->url_google_sheet_view,
            "user_id"               => auth()->user()->id,
            "supplies_cost"         => $req->supplies_cost ?? 0,
            "class_cost"            => $req->class_cost ?? 0,
            "total_cost"            => $req->total_cost ?? 0,
        ];
        $req->validate([
            "class_status_id"   => "required",
            "class_level_id"    => "required",
            "class_type_id"     => "required",
            "staff_id"          => "required",
            "shift_id"          => "required",
            "class_code"        => "required|unique:classes,class_code" . ($id ? ",$id" : ''),
            "generation"        => "required",
            "group"             => "required",
            "period_duration"   => "required",
            "start_date"        => "required",
            "end_date"          => "required",
            'class_cost'        => 'nullable|numeric',
            'supplies_cost'     => 'nullable|numeric',
            'total_cost'        => 'nullable|numeric',
        ], [
            "class_status_id.required"  => "Please input this field",
            "class_level_id.required"   => "Please input this field",
            "class_type_id.required"    => "Please input this field",
            "staff_id.required"         => "Please input this field",
            "shift_id.required"         => "Please input this field",
            "class_code.required"       => "Please input this field",
            "generation.required"       => "Please input this field",
            "group.required"            => "Please input this field",
            "period_duration.required"  => "Please input this field",
            "start_date.required"       => "Please input this field",
            "end_date.required"         => "Please input this field",
            "class_code.numeric"        => "Please input number",
            "supplies_code.numeric"     => "Please input number",
            "total_code.numeric"        => "Please input number",
        ]);
        $status = "Create success.";
        DB::beginTransaction();
        try {
            if (!$id) {
                Classes::create($item);
            } else {
                Classes::find($id)->update($item);
                $status = "Update success.";
            }
            DB::commit();
            Session::flash("success", $status);
            return redirect()->route("admin-class-list", 1);
        } catch (Exception $error) {
            dd($error);
            DB::rollback();
            Session::flash('warning', 'Create unsuccess!');
            return redirect()->back();
        }
    }

    public function onView(Request $req)
    {

        // update or create student attendance status
        // (new AttendanceStatusController)->setIsAttendancePaste($req->id);

        $data['data'] = Classes::find($req->id);
        $data['student_in_class'] = Classes::find($req->id)->students()->with('exam_score', function ($q) use ($req) {
            $q->where('class_id', $req->id);
        })
            ->with('student_score', function ($q) use ($req) {
                $q->where('class_id', $req->id);
            })
            ->with('student_intern_places', function ($q) use ($req) {
                $q->where('class_id', $req->id);
            })
            ->with('attendance_statuses', function ($q) use ($req) {
                $q->where('class_id', $req->id);
            })
            ->get();
        $data['student_intern'] = StudentInternPlace::where('class_id', $req->id)->with('student', 'internship_place', 'classes')->get();
        $lesson_id = [];
        if ($data['data']->class_structure_program) {
            $lesson_id = $data['data']->class_structure_program->structure_program->lesson_id;
        }
        $data['lessons'] = Lesson::whereIn('id', $lesson_id)->get();
        $data['limit_3_lessons'] = Lesson::whereIn('id', $lesson_id)->take(3)->get();
        $data['attendances'] = Attendance::where('class_id', $req->id)->get();


        // check student attendance everyday
        $data['today'] = Carbon::now()->format('d/M/Y');
        $data['today_attend'] = Attendance::where('class_id', $req->id)->whereDate('log_date', Carbon::now()->format('Y/m/d'))->where('is_attended', 1)->latest()->first();

        // get student study group
        $data['groups'] = Group::where('class_id', $req->id)->with('students')->get();

        // student attendance status
        $data['attendance_status'] = (new AttendanceStatusController)->getIsAttendancePaste($req->id);
        // total attendance
        // $data['total_attendance'] = (new AttendanceStatusController)->calcClassDuration($req->id);

        return view($this->layout . 'view-info', $data);
    }

    public function onAssignStructured(Request $req)
    {

        $data['data'] = Classes::find($req->id);
        $data['structured_programs'] = StructureProgram::orderByDesc("id")->get();
        $data['lessons'] = Lesson::all();
        if ($data['data']->class_structure_programs) {
            return redirect()->route("admin-class-list", 1);
        } else {
            return view($this->layout . 'assign-structured-program', $data);
        }
    }

    public function onSaveAssignStructured(Request $req)
    {
        $item = [
            'class_id' => $req->id,
            'structure_program_id' => $req->structure_program_id,
        ];
        DB::beginTransaction();
        try {
            ClassStructureProgram::create($item);
            DB::commit();
            Session::flash("success", 'Assign structured program success');
            return redirect()->route("admin-class-view-info", $req->id);
        } catch (Exception $error) {
            DB::rollback();
            Session::flash('warning', 'Assign structured program unsuccess!');
            return redirect()->back();
        }
    }

    public function onUpdateStatus(Request $req)
    {
        $status = true;
        $item = [
            "status" => $req->status,
        ];
        DB::beginTransaction();
        try {
            $status = $req->status == 2 ? "Disable successful!" : "Enable successful!";
            Classes::where("id", $req->id)->update($item);
            DB::commit();
            Session::flash("success", $status);
        } catch (Exception $error) {
            Session::flash('warning', 'Create unsuccess!');
        }
        return redirect()->back();
    }

    public function studyMaterial()
    {
        $data['data'] = Classes::findOrFail(request('class_id'))->students()->get();
        return view($this->layout . 'studyMaterial.index', $data);
    }

    public function studentCard()
    {
        if (!request('student_id') || !request('class_id')) {
            Session::flash("warning", "Invalided valued");
            return redirect()->back();
        }
        $data['data'] = Student::whereIn('id', request('student_id'))->get();
        return view($this->layout . 'studyMaterial.student-card', $data);
    }

    public function onRemoveStudent()
    {
        $class = Classes::findorFail(request('class_id'));
        DB::beginTransaction();
        try {
            $class->Students()->detach(request('student_id'));
            DB::commit();
            Session::flash("success", 'Remove student success');
        } catch (Exception $error) {
            DB::rollBack();
            Session::flash('warning', 'Remove student unsuccess!');
        }
        return redirect()->back();
    }

    public function onAllowUser(Request $req)
    {
        $data['data'] = Classes::find($req->id)->users()->get();
        $data['class_allow_user_id'] = [];
        foreach ($data['data'] as $item) {
            $data['class_allow_user_id'][] = $item->id;
        }
        $data['users'] = User::where('status', 1)->where('role', '!=', config('dummy.user.role.super_admin'))->get();
        return view($this->layout . 'allow-user', $data);
    }

    public function onSaveAllowUser(Request $req)
    {
        $data = Classes::find($req->id);
        DB::beginTransaction();
        try {
            $data->users()->sync($req->user_id);
            DB::commit();
            Session::flash("success", 'Assign success');
            return redirect()->route('admin-class-list', 1);
        } catch (Exception $error) {
            dd($error);
            DB::rollback();
            Session::flash('warning', 'Assign unsuccess!');
            return redirect()->back();
        }
    }
}
