<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Attendance;
use App\Models\Lesson;
use App\Models\Classes;
use App\Models\ClassStructureProgram;
use App\Models\Student;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Session;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Commands\Show;

class AttendanceController extends Controller
{
    protected $layout = 'admin::pages.attendance.';
    public function __construct()
    {
        $this->middleware('permission:attendance-view', ['only' => ['index']]);
        $this->middleware('permission:attendance-create', ['only' => ['onCreate', 'onSave']]);
        $this->middleware('permission:attendance-update', ['only' => ['onCreate', 'onSave', 'onUpdateStatus']]);
    }
    public function index(Request $req)
    {
        if (!$req->id) {
            return redirect()->route('admin-class-list', 1);
        }
        $data['data'] = Attendance::when(filled(request('keyword')), function ($q) {
            $q->where(function ($q) {
                $q->where('class_code', 'like', '%' . request('keyword') . '%');
            });
        })
            ->orderByDesc("id")
            ->paginate(50);
        return view($this->layout . 'index', $data);
    }

    public function onCreate(Request $req)
    {
        $classes = Classes::find($req->class_id);
        $csp = ClassStructureProgram::where('class_id', $req->class_id)->first();
        $data['lessons'] = Lesson::all();
        $data['students'] = [];
        $data['lessons_id'] = [];
        if (isset($csp->structure_program)) {
            $data['lessons_id'] = $csp->structure_program->lesson_id;
        }
        if (isset($classes->students)) {
            $data['students'] = $classes->students;
        }
        return view($this->layout . 'create', $data);
    }

    public function onSave(Request $req)
    {
        $id = $req->id;
        $item = [
            "class_id" => $req->class_id,
            "lesson_id" => $req->lesson_id,
            "student_id" => array_map('intval', $req->student_id),
            "is_attended" => 1,
            "log_date" => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        $status = "Create success.";
        DB::beginTransaction();
        try {
            if (!$id) {
                Attendance::create($item);
            } else {
                Attendance::find($id)->update($item);
                $status = "Update success.";
            }
            // update or create student attendance status
            (new AttendanceStatusController)->setIsAttendancePaste($req->class_id);
            DB::commit();
            Session::flash("success", $status);
            return redirect()->route("admin-class-view-info", $req->class_id);
        } catch (Exception $error) {
            dd($error);
            DB::rollback();
            Session::flash('warning', 'Create unsuccess!');
            return redirect()->back();
        }
    }

    public function onUpdateStudentAttendance()
    {
        if (empty(request('lesson_id')) || empty(request('class_id')) || empty(request('student_id')) || empty(request('log_date'))) {
            return response()->json([
                "code" => 40005,
                "message" => "Invalided value",
            ]);
        } else {
            $log_date = (new Carbon(request('log_date')))->format('Y-m-d');

            $data = Attendance::whereDate('log_date', $log_date)->where('lesson_id', request('lesson_id'))->where('class_id', request('class_id'))->first();
            $student = Student::findOrFail(request('student_id'));
            $class_id = [];
            foreach ($student->classes as $item) {
                $class_id[] = $item->id;
            }
            if (empty($data) || $data == null) {
                return response()->json([
                    "code" => 40005,
                    "message" => "No matching data",
                ]);
            } else {
                $student = $data->student_id;
                DB::beginTransaction();
                try {
                    if (!in_array(request('student_id'), $student) && in_array(request('class_id'), $class_id)) {
                        $student[] = (int)request('student_id');
                        $data->update([
                            'student_id' => $student,
                        ]);
                        // update or create student attendance status
                        (new AttendanceStatusController)->setIsAttendancePaste(request('class_id'));
                        DB::commit();
                        return response()->json([
                            "code" => 20000,
                            "message" => "Update student attendance success",
                        ]);
                    } else {
                        return response()->json([
                            "code" => 40001,
                            "message" => "Student already have attendance or Student is not in the current class",
                        ]);
                    }
                } catch (Exception $e) {
                    DB::rollback();
                    return response()->json([
                        "code" => 40005,
                        "message" => "Update Student unsuccess!",
                    ]);
                }
            }
        }
    }

    public function onDeleteStudentAttendance()
    {

        $data = Attendance::findOrFail(request('id'));
        DB::beginTransaction();
        try {
            if (in_array(request('student_id'), $data->student_id)) {
                $students = $data->student_id;
                foreach ($students as $index => $student) {
                    if ($student == request('student_id')) {
                        unset($students[$index]);
                    }
                }
                $student_id = array_values($students);
                $data->update([
                    'student_id' => $student_id,
                ]);
                // update or create student attendance status
                (new AttendanceStatusController)->setIsAttendancePaste($data->class_id);
                DB::commit();
                return response()->json([
                    "code" => 20000,
                    "message" => "Delete student attendance success."
                ]);
            } else {
                return response()->json([
                    "code" => 40005,
                    "message" => "Student is not in the current class",
                ]);
            }
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                "code" => 40005,
                "message" => "Delete student attendance unsuccess!",
            ]);
        }
    }
    
    public function onDeleteAttendance(Request $req)
    {
        $id = $req->id;
        $data = Attendance::find($id);
        DB::beginTransaction();
        try {
            $data->delete();
            // update or create student attendance status
            (new AttendanceStatusController)->setIsAttendancePaste($data->class_id);
            DB::commit();
            Session::flash('success', 'Deleted success.');
            return redirect()->back();
        } catch (Exception $e) {
            DB::rollback();
            Session::flash('warning', "Deleted unseccess!");
            return redirect()->back();
        }
    }
}
